/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2006 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_log.c 9258 2006-09-05 13:24:55Z halr $
 */


/*
 * Abstract:
 *    Implementaion of osm_log_t.
 * This object represents the log file.
 * This object is part of the opensm family of objects.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.8 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <opensm/osm_log.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#ifndef WIN32
#include <sys/time.h>
#include <unistd.h>
#include <complib/cl_timer.h>

static char *month_str[] = {
  "Jan",
  "Feb",
  "Mar",
  "Apr",
  "May",
  "Jun",
  "Jul",
  "Aug",
  "Sep",
  "Oct",
  "Nov",
  "Dec"
};
#endif /* ndef WIN32 */

static int log_exit_count = 0;

static void truncate_log_file(osm_log_t* const p_log)
{
	int fd = fileno(p_log->out_port);
	if (ftruncate(fd, 0) < 0)
		fprintf(stderr, "truncate_log_file: cannot truncate: %s\n",
			strerror(errno));
	if (lseek(fd, 0, SEEK_SET) < 0)
		fprintf(stderr, "truncate_log_file: cannot rewind: %s\n",
			strerror(errno));
	p_log->count = 0;
}

void
osm_log(
  IN osm_log_t* const p_log,
  IN const osm_log_level_t verbosity,
  IN const char *p_str, ... )
{
  char      buffer[LOG_ENTRY_SIZE_MAX];
  va_list   args;
  int       ret;

#ifdef WIN32
  SYSTEMTIME st;
  uint32_t pid = GetCurrentThreadId();
#else
  pid_t pid = 0;
  time_t tim;
  struct tm result;
  uint64_t time_usecs;
  uint32_t usecs;

  time_usecs = cl_get_time_stamp();
  tim = time_usecs/1000000;
  usecs = time_usecs % 1000000;
  localtime_r(&tim, &result);
#endif /* WIN32 */

  /* If this is a call to syslog - always print it */
  if ( verbosity & (OSM_LOG_SYS | p_log->level) )
  {
    va_start( args, p_str );
    vsprintf( buffer, p_str, args );
    va_end(args);

    /* this is a call to the syslog */
    if (verbosity & OSM_LOG_SYS)
    {
      cl_log_event("OpenSM", LOG_INFO, buffer , NULL, 0);

      /* SYSLOG should go to stdout too */
      if (p_log->out_port != stdout)
      {
        printf("%s\n", buffer);
        fflush( stdout );
      }
    }

    /* regular log to default out_port */
    cl_spinlock_acquire( &p_log->lock );

    if (p_log->max_size && p_log->count > p_log->max_size)
    {
      /* truncate here */
      fprintf(stderr, "osm_log: log file exceeds the limit %lu. Truncating.\n",
              p_log->max_size);
      truncate_log_file(p_log);
    }

#ifdef WIN32
    GetLocalTime(&st);
 _retry:
    ret = fprintf( p_log->out_port, "[%02d:%02d:%02d:%03d][%04X] -> %s",
                   st.wHour, st.wMinute, st.wSecond, st.wMilliseconds,
                   pid, buffer );
#else
    pid = pthread_self();
 _retry:
    ret = fprintf( p_log->out_port, "%s %02d %02d:%02d:%02d %06d [%04X] -> %s",
                   (result.tm_mon < 12 ? month_str[result.tm_mon] : "???"),
                   result.tm_mday, result.tm_hour,
                   result.tm_min, result.tm_sec,
                   usecs, pid, buffer );
#endif
 
    /*  flush log */
    if (ret > 0 && (p_log->flush || (verbosity & OSM_LOG_ERROR)) &&
        fflush( p_log->out_port ) < 0)
      ret = -1;

    if (ret >= 0)
    {
      log_exit_count = 0;
      p_log->count += ret;
    }
    else if (log_exit_count < 3)
    {
      log_exit_count++;
      if (errno == ENOSPC && p_log->max_size) {
        fprintf(stderr, "osm_log: write failed: %s. Truncating log file.\n",
                strerror(errno));
        truncate_log_file(p_log);
        goto _retry;
      }
      fprintf(stderr, "osm_log: write failed: %s\n", strerror(errno));
    }

    cl_spinlock_release( &p_log->lock );
  }
}

void
osm_log_raw(
  IN osm_log_t* const p_log,
  IN const osm_log_level_t verbosity,
  IN const char *p_buf )
{
  if( p_log->level & verbosity )
  {
    cl_spinlock_acquire( &p_log->lock );
    printf( "%s", p_buf );
    cl_spinlock_release( &p_log->lock );

    /*
      Flush log on errors too.
    */
    if( p_log->flush || (verbosity & OSM_LOG_ERROR) )
      fflush( stdout );
  }
}

boolean_t
osm_is_debug(void)
{
#if defined( _DEBUG_ )
  return TRUE;
#else
  return FALSE;
#endif /* defined( _DEBUG_ ) */
}

ib_api_status_t
osm_log_init_v2(
  IN osm_log_t* const p_log,
  IN const boolean_t flush,
  IN const uint8_t log_flags,
  IN const char *log_file,
  IN const unsigned long max_size,
  IN const boolean_t accum_log_file )
{
  struct stat st;

  p_log->level = log_flags;
  p_log->flush = flush;
  p_log->count = 0;
  p_log->max_size = 0;

  if (log_file == NULL || !strcmp(log_file, "-") ||
      !strcmp(log_file, "stdout"))
  {
    p_log->out_port = stdout;
  }
  else if (!strcmp(log_file, "stderr"))
  {
    p_log->out_port = stderr;
  }
  else
  {
    if (accum_log_file)
      p_log->out_port = fopen(log_file, "a+");
    else
      p_log->out_port = fopen(log_file, "w+");

    if (!p_log->out_port)
    {
      if (accum_log_file)
        printf("Cannot open %s for appending. Permission denied\n", log_file);
      else
        printf("Cannot open %s for writing. Permission denied\n", log_file);

      return(IB_UNKNOWN_ERROR);
    }

    if (fstat(fileno(p_log->out_port), &st) == 0)
      p_log->count = st.st_size;

    p_log->max_size = max_size;
  }

  openlog("OpenSM", LOG_CONS | LOG_PID, LOG_USER);

  if (cl_spinlock_init( &p_log->lock ) == CL_SUCCESS) 
    return IB_SUCCESS;
  else
    return IB_ERROR;
}

ib_api_status_t
osm_log_init(
  IN osm_log_t* const p_log,
  IN const boolean_t flush,
  IN const uint8_t log_flags,
  IN const char *log_file,
  IN const boolean_t accum_log_file )
{
  return osm_log_init_v2( p_log, flush, log_flags, log_file, 0, accum_log_file );
}
